function validStiffP1()
% function validStiffP1()
%  Validation function for the assembly of the stiffness matrix for
%  `P_1`-Lagrange finite element method
%
%   The Stiffness Matrix `\Stiff` is given by 
%   ``\Stiff_{i,j}=\int_\DOMH \DOT{\GRAD\FoncBase_i(\q)}{\GRAD\FoncBase_j(\q)}d\q,\ \forall (i,j)\in\ENS{1}{\nq}^2``
%   where `\FoncBase_i` are `P_1`-Lagrange basis functions.
%   This Matrix is computed by functions StiffAssemblingP1{Version} where {Version} is one of
%     'base', 'OptV0', 'OptV1' and 'OptV2'.
%     - Test 1: Computation of the Stiffness Matrix using all the versions giving errors and cputimes
%     - Test 2: Computation of the integral ``\int_\DOM \DOT{\GRAD u(\q)}{\GRAD v(\q)}d\q \approx \DOT{\Stiff \vecb{U}}{\vecb{V}}``
%       where `\vecb{U}_i=u(\q^i)` and `\vecb{V}_i=v(\q^i)`.
%       Functions `u` and `v` are those defined in #valid_FEMmatrices.
%     - Test 3: One retrieves the order 2 of `P_1`-Lagrange integration 
%       ``|\int_\DOM \DOT{\GRAD u}{\GRAD v} -\DOT{\GRAD \Pi_h(u)}{\GRAD \Pi_h(v)}d\DOM| \leq C h^2``
%
% See also:
%   #StiffAssemblingP1base, #StiffAssemblingP1OptV0,
%   #StiffAssemblingP1OptV1, #StiffAssemblingP1OptV2,
%   #valid_FEMmatrices, #SquareMesh, #GetMaxLengthEdges
% 
% @author François Cuvelier @date 2012-11-26
%
% OptFEM2DP1 [V1.0e] -    Copyright (C) 2013  CJS (LAGA)
%
%   This file is part of OptFEM2DP1.
%   OptFEM2DP1 is free software: you can redistribute it and/or modify
%   it under the terms of the GNU General Public License as published by
%   the Free Software Foundation, either version 3 of the License, or
%   (at your option) any later version.
%
%   OptFEM2DP1 is distributed in the hope that it will be useful,
%   but WITHOUT ANY WARRANTY; without even the implied warranty of
%   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%   GNU General Public License for more details.
%
%   You should have received a copy of the GNU General Public License
%   along with this program.  If not, see <http://www.gnu.org/licenses/>.

  disp('*******************************************')
  disp('*     Stiff Assembling P1 validations     *')
  disp('*******************************************')

  Th=SquareMesh(50);

% TEST 1
  disp('-----------------------------------------')
  disp('  Test 1: Matrices errors and CPU times  ')
  disp('-----------------------------------------')
  tic();
  Mbase=StiffAssemblingP1base(Th.nq,Th.nme,Th.q,Th.me,Th.areas);
  T(1)=toc();
  tic();
  MOptV0=StiffAssemblingP1OptV0(Th.nq,Th.nme,Th.q,Th.me,Th.areas);
  T(2)=toc();
  Test1.error(1)=norm(Mbase-MOptV0,Inf);
  Test1.name{1}='StiffAssemblingP1OptV0';
  fprintf('    Error P1base vs OptV0 : %e\n',Test1.error(1))
  tic();
  MOptV1=StiffAssemblingP1OptV1(Th.nq,Th.nme,Th.q,Th.me,Th.areas);
  T(3)=toc();
  Test1.error(2)=norm(Mbase-MOptV1,Inf);
  Test1.name{2}='StiffAssemblingP1OptV1';
  fprintf('    Error P1base vs OptV1 : %e\n',Test1.error(2))
  tic();
  MOptV2=StiffAssemblingP1OptV2(Th.nq,Th.nme,Th.q,Th.me,Th.areas);
  T(4)=toc();
  Test1.error(3)=norm(Mbase-MOptV2,Inf);
  Test1.name{3}='StiffAssemblingP1OptV2';
  fprintf('    Error P1base vs OptV2 : %e\n',Test1.error(3))

  fprintf('    CPU times base (ref) : %3.4f (s)\n',T(1))
  fprintf('    CPU times OptV0       : %3.4f (s) - Speed Up X%3.3f\n',T(2),T(1)/T(2))
  fprintf('    CPU times OptV1       : %3.4f (s) - Speed Up X%3.3f\n',T(3),T(1)/T(3))
  fprintf('    CPU times OptV2       : %3.4f (s) - Speed Up X%3.3f\n',T(4),T(1)/T(4))
  checkTest1(Test1)

  M=Mbase;

% TEST 2
  disp('-----------------------------------------------------')
  disp('  Test 2: Validations by integration on [0,1]x[0,1]  ')
  disp('-----------------------------------------------------')
  Test=valid_FEMmatrices();
  for kk=1:length(Test)
    U=Test(kk).u(Th.q(1,:),Th.q(2,:));
    V=Test(kk).v(Th.q(1,:),Th.q(2,:));
    Test(kk).error=abs(Test(kk).Stiff-U*M*V');
    fprintf('    functions %d : u(x,y)=%s, v(x,y)=%s,\n           -> Stiff error=%e\n',kk,Test(kk).cu,Test(kk).cv,Test(kk).error);
  end
  checkTest2(Test)
  
% TEST 3
  disp('--------------------------------')
  disp('  Test 3: Validations by order  ')
  disp('--------------------------------')
  n=length(Test);
  u=Test(n).u;
  v=Test(n).v;
  ExSol=Test(n).Stiff;

  for k=1:10  
    Th=SquareMesh(50*k+50);
    fprintf('    Matrix size : %d\n',Th.nq);
    h(k)=GetMaxLengthEdges(Th.q,Th.me);
    tic();
    M=StiffAssemblingP1OptV2(Th.nq,Th.nme,Th.q,Th.me,Th.areas);
    TT(k)=toc();
    U=u(Th.q(1,:),Th.q(2,:));
    V=v(Th.q(1,:),Th.q(2,:));
    Error(k)=abs(ExSol-U*M*V');
    fprintf('      StiffAssemblingP1OptV2 CPU times : %3.3f(s)\n',TT(k));
    fprintf('      Error                            : %e\n',Error(k));
  end

  loglog(h,Error,'+-r',h,h*1.1*Error(1)/h(1),'-sm',h,1.1*Error(1)*(h/h(1)).^2,'-db')
  legend('Error','O(h)','O(h^2)')
  xlabel('h')
  title('Test 3 : Stiffness Matrix')
  checkTest3(h,Error)
end

function checkTest1(Test)
  I=find(Test.error>1e-14);
  if isempty(I)
    disp('------------------------')
    disp('  Test 1 (results): OK')
    disp('------------------------')
  else
    disp('----------------------------')
    disp('  Test 1 (results): FAILED')
    disp('----------------------------')
  end
end

function checkTest2(Test)
  N=length(Test);
  cntFalse=0;
  for k=1:N
    if (Test(k).degree<=1)
      if (Test(k).error>1e-12)
        cntFalse=cntFalse+1;
      end
    end
  end
  if (cntFalse==0)
    disp('------------------------')
    disp('  Test 2 (results): OK')
    disp('------------------------')
  else
    disp('----------------------------')
    disp('  Test 2 (results): FAILED')
    disp('----------------------------')
  end
end

function checkTest3(h,error)
  % order 2
  P=polyfit(log(h),log(error),1);
  if abs(P(1)-2)<1e-2
    disp('------------------------')
    disp('  Test 3 (results): OK')
    fprintf('    -> found numerical order %f. Must be 2\n',P(1))
    disp('------------------------')
  else
    disp('----------------------------')
    disp('  Test 3 (results): FAILED')
    fprintf('    -> found numerical order %f. Must be 2\n',P(1))
    disp('----------------------------')
  end
end
