# From Luke Olson (http://lukeo.cs.illinois.edu/)
#   https://github.com/lukeolson/python-mesh-scripts.git
# gmsh reader
# neu writer
#
# * handles triangles (2d), tets(3d)
import numpy
#import scipy
#from scipy.sparse import lil_matrix, csr_matrix, coo_matrix, triu, eye
import sys


class gmshMesh:
    """
    Store the verts and elements and physical data

    attributes
    ----------
    Verts : array
        array of 3d coordinates (npts x 3)
    Elmts : dict
        dictionary of tuples
        (rank 1 array of physical ids, rank 2 array of element to vertex ids
        (Nel x ppe)) each array in the tuple is of length nElmts Phys : dict
        keys and names

    methods
    -------
    read_msh:
        read a 2.0 ascii gmsh file
    write_neu:
        write a gambit neutral file. works for tets, tris in 3d and 2d
    write_vtu:
        write VTK file (calling vtk_writer.py)
    """
    def __init__(self):

        self.Verts = []
        self.Elmts = {}
        self.Phys = {}

        self.npts = 0
        self.nElmts = {}
        self.nprops = 0

        self._elm_types()   # sets elm_type

        self.meshname = ""

    def read_msh(self, mshfile):
        """Read a Gmsh .msh file.

        Reads Gmsh 2.0 mesh files
        """
        self.meshname = mshfile
        try:
            fid = open(mshfile, "r")
        except IOError:
            print("File '%s' not found." % (filename))
            sys.exit()

        line = 'start'
        while line:
            line = fid.readline()

            if line.find('$MeshFormat') == 0:
                line = fid.readline()
                if line.split()[0][0] is not '2':
                    print("wrong gmsh version")
                    sys.exit()
                line = fid.readline()
                if line.find('$EndMeshFormat') != 0:
                    raise ValueError('expecting EndMeshFormat')

            if line.find('$PhysicalNames') == 0:
                line = fid.readline()
                self.nprops = int(line.split()[0])
                for i in range(0, self.nprops):
                    line = fid.readline()
                    newkey = int(line.split()[0])
                    qstart = line.find('"')+1
                    qend = line.find('"', -1, 0)-1
                    self.Phys[newkey] = line[qstart:qend]
                line = fid.readline()
                if line.find('$EndPhysicalNames') != 0:
                    raise ValueError('expecting EndPhysicalNames')

            if line.find('$Nodes') == 0:
                line = fid.readline()
                self.npts = int(line.split()[0])
                self.Verts = numpy.zeros((self.npts, 3), dtype=float)
                for i in range(0, self.npts):
                    line = fid.readline()
                    data = line.split()
                    idx = int(data[0])-1  # fix gmsh 1-based indexing
                    if i != idx:
                        raise ValueError('problem with vertex ids')
                    self.Verts[idx, :] = list(map(float, data[1:])) # FC : fixe to run with Python3
                line = fid.readline()
                if line.find('$EndNodes') != 0:
                    raise ValueError('expecting EndNodes')

            if line.find('$Elements') == 0:
                line = fid.readline()
                self.nel = int(line.split()[0])
                for i in range(0, self.nel):
                    line = fid.readline()
                    data = line.split()
                    idx = int(data[0])-1  # fix gmsh 1-based indexing
                    if i != idx:
                        raise ValueError('problem with elements ids')
                    etype = int(data[1])           # element type
                    nnodes = self.elm_type[etype]   # lookup number of nodes
                    ntags = int(data[2])           # number of tags following
                    k = 3
                    if ntags > 0:                   # set physical id
                        physid = int(data[k])
                        if physid not in self.Phys:
                            self.Phys[physid] = 'Physical Entity %d' % physid
                            self.nprops += 1
                        k += ntags

                    verts = list(map(int, data[k:])) # FC : fixe to run with Python3
                    verts = numpy.array(verts)-1  # fixe gmsh 1-based index

                    if (etype not in self.Elmts) or\
                            (len(self.Elmts[etype]) == 0):
                        # initialize
                        self.Elmts[etype] = (physid, verts)
                        self.nElmts[etype] = 1
                    else:
                        # append
                        self.Elmts[etype] = \
                            (numpy.hstack((self.Elmts[etype][0], physid)),
                             numpy.vstack((self.Elmts[etype][1], verts)))
                        self.nElmts[etype] += 1

                line = fid.readline()
                if line.find('$EndElements') != 0:
                    raise ValueError('expecting EndElements')
        fid.close()

    def _find_EF(self, vlist, E):
        for i in range(0, E.shape[0]):
            enodes = E[i, :]
            if len(numpy.intersect1d_nu(vlist, enodes)) == len(vlist):
                # found element.  now the face
                missing_node = numpy.setdiff1d(enodes, vlist)
                loc = numpy.where(enodes == missing_node)[0][0]

                # determine face from missing id
                if len(enodes) == 3:  # tri
                    face_map = {0: 1, 1: 2, 2: 0}
                if len(enodes) == 4:  # tet
                    face_map = {0: 2, 1: 3, 2: 1, 3: 0}

                return i, face_map[loc]

    def _elm_types(self):
        elm_type = {}
        elm_type[1] = 2    # 2-node line
        elm_type[2] = 3    # 3-node triangle
        elm_type[3] = 4    # 4-node quadrangle
        elm_type[4] = 4    # 4-node tetrahedron
        elm_type[5] = 8    # 8-node hexahedron
        elm_type[6] = 6    # 6-node prism
        elm_type[7] = 5    # 5-node pyramid
        elm_type[8] = 3    # 3-node second order line
                            # (2 nodes at vertices and 1 with edge)
        elm_type[9] = 6    # 6-node second order triangle
                            # (3 nodes at vertices and 3 with edges)
        elm_type[10] = 9    # 9-node second order quadrangle
                            # (4 nodes at vertices,
                            #  4 with edges and 1 with face)
        elm_type[11] = 10   # 10-node second order tetrahedron
                            # (4 nodes at vertices and 6 with edges)
        elm_type[12] = 27   # 27-node second order hexahedron
                            # (8 nodes at vertices, 12 with edges,
                            #  6 with faces and 1 with volume)
        elm_type[13] = 18   # 18-node second order prism
                            # (6 nodes at vertices,
                            #  9 with edges and 3 with quadrangular faces)
        elm_type[14] = 14   # 14-node second order pyramid
                            # (5 nodes at vertices,
                            #  8 with edges and 1 with quadrangular face)
        elm_type[15] = 1    # 1-node point
        elm_type[16] = 8    # 8-node second order quadrangle
                            # (4 nodes at vertices and 4 with edges)
        elm_type[17] = 20   # 20-node second order hexahedron
                            # (8 nodes at vertices and 12 with edges)
        elm_type[18] = 15   # 15-node second order prism
                            # (6 nodes at vertices and 9 with edges)
        elm_type[19] = 13   # 13-node second order pyramid
                            # (5 nodes at vertices and 8 with edges)
        elm_type[20] = 9    # 9-node third order incomplete triangle
                            # (3 nodes at vertices, 6 with edges)
        elm_type[21] = 10   # 10-node third order triangle
                            # (3 nodes at vertices, 6 with edges, 1 with face)
        elm_type[22] = 12   # 12-node fourth order incomplete triangle
                            # (3 nodes at vertices, 9 with edges)
        elm_type[23] = 15   # 15-node fourth order triangle
                            # (3 nodes at vertices, 9 with edges, 3 with face)
        elm_type[24] = 15   # 15-node fifth order incomplete triangle
                            # (3 nodes at vertices, 12 with edges)
        elm_type[25] = 21   # 21-node fifth order complete triangle
                            # (3 nodes at vertices, 12 with edges, 6 with face)
        elm_type[26] = 4    # 4-node third order edge
                            # (2 nodes at vertices, 2 internal to edge)
        elm_type[27] = 5    # 5-node fourth order edge
                            # (2 nodes at vertices, 3 internal to edge)
        elm_type[28] = 6    # 6-node fifth order edge
                            # (2 nodes at vertices, 4 internal to edge)
        elm_type[29] = 20   # 20-node third order tetrahedron
                            # (4 nodes at vertices, 12 with edges,
                            #  4 with faces)
        elm_type[30] = 35   # 35-node fourth order tetrahedron
                            # (4 nodes at vertices, 18 with edges,
                            #  12 with faces, 1 in volume)
        elm_type[31] = 56   # 56-node fifth order tetrahedron
                            # (4 nodes at vertices, 24 with edges,
                            #  24 with faces, 4 in volume)
        self.elm_type = elm_type


if __name__ == '__main__':

    #meshname = 'test.msh'
    meshname = 'condenser01.msh'
    mesh = gmshMesh()
    mesh.read_msh(meshname)
    print(mesh.Elmts[2][1].shape)

    import trimesh
    trimesh.trimesh(mesh.Verts[:, :2], mesh.Elmts[2][1])
    import pylab
    pylab.plot(mesh.Verts[mesh.bid, 0], mesh.Verts[mesh.bid, 1], 'ro')
    if 0:
        import trimesh
        trimesh.trimesh(mesh.Verts[:, :2], mesh.Elmts[2][1])

        mesh.refine2dtri([0, 3, 5])
        trimesh.trimesh(mesh.Verts[:, :2], mesh.Elmts[2][1])

        mesh.refine2dtri()
        mesh.refine2dtri()
        mesh.refine2dtri()
        trimesh.trimesh(mesh.Verts[:, :2], mesh.Elmts[2][1])
        print(mesh.Elmts[2][1].shape)

    #mesh.refine2dtri(marked_elements=[0, 3, 5])

    import pylab
    import demo
    #demo.trimesh(mesh.Verts[:, 0:2],mesh.Elmts[2][1])

  # mesh.write_vtu()
    #mesh.write_neu(bdy_ids=4)
  #  mesh.write_neu()
