import matplotlib.pyplot as plt
import numpy as np
import sys,os
sep=os.path.sep
ToolBoxDir=os.path.abspath(os.path.dirname(os.path.abspath(__file__))+sep+"..")
sys.path.append(ToolBoxDir)

from pyVecFEMP1Light.mesh import readFreeFEM
from pyVecFEMP1Light.pde import solvePDE,initPDE,setBC_PDE,buildPDEsystem,splitPDEsol
from pyVecFEMP1Light.FEMtools import setFdata,getVFindices
from pyVecFEMP1Light.operators import Loperator, Hoperator
from pyVecFEMP1Light.paraview import vtkWriteValues
from pyVecFEMP1Light.graphics import PlotVal,PlotMesh,PlotBounds,showSparsity
from pyVecFEMP1Light.common import mkdir_p,pause
from scipy.sparse.linalg import spsolve

plt.close("all")

MeshDir=ToolBoxDir+sep+'meshes'+sep+'2D'
Name='HeatAndFlowVelocity2D01'
MeshFile=MeshDir+sep+'FlowVelocity2d01-50.msh'
VTKsaveDir=ToolBoxDir+sep+'vtk'

d=2

print('1. Get mesh %s using pyVecFEMP1Light'%MeshFile)
Th=readFreeFEM(MeshFile)
print('  -> Mesh sizes : nq=%d, nme=%d, nbe=%d'%(Th.nq,Th.nme,Th.nbe));

print('2. Set 2D potential flow BVP problem')
Hop=Hoperator(d=d,m=3)
Hop.H[0][1]=Loperator(d=d,b=[-1,0])
Hop.H[0][2]=Loperator(d=d,b=[0,-1])
Hop.H[1][0]=Loperator(d=d,c=[-1,0])
Hop.H[1][1]=Loperator(d=d,a0=1)
Hop.H[2][0]=Loperator(d=d,c=[0,-1])
Hop.H[2][2]=Loperator(d=d,a0=1)
pdeFlow=initPDE(Hop,Th)

pdeFlow=setBC_PDE(pdeFlow,20,0,'Dirichlet',20)
pdeFlow=setBC_PDE(pdeFlow,21,0,'Dirichlet',-20)

print('3.a Build 2D potential flow linear system')
num=1 # num=1, block basis - num=0, alternate basis
Apf,bpf,IDpf,IDcpf,gDpf=buildPDEsystem(pdeFlow,Num=num) 

print('3.b Solve 2D potential flow linear system')
X=np.zeros((pdeFlow.m*Th.nq,))
X[IDpf]=gDpf[IDpf]
bb=bpf[IDcpf]-Apf[IDcpf,::]*gDpf;
X[IDcpf]=spsolve((Apf[IDcpf])[::,IDcpf],bb)
W=splitPDEsol(pdeFlow,X,num)

print('4. Set 2D stationary heat PDE with potential flow')

af=lambda x,y: 0.1+y**2;
Lop=Loperator(d=Th.d,A=[[af,None],[None,af]],c=[W[1],W[2]],a0=0.01);

pde=initPDE(Lop,Th)
pde=setBC_PDE(pde,21,0,'Dirichlet',  lambda x,y: 20*y )
pde=setBC_PDE(pde,22,0,'Dirichlet',  0)
pde=setBC_PDE(pde,23,0,'Dirichlet',  0)

print('5.a Build 2D stationary heat linear system with potential flow')
A,b,ID,IDc,gD=buildPDEsystem(pde)

print('5.b Solve 2D stationary heat linear system with potential flow')
u=np.zeros((Th.nq,))
u[ID]=gD[ID]
bb=b[IDc]-A[IDc,::]*gD;
u[IDc]=spsolve((A[IDc])[::,IDc],bb)

print('6. Graphic representations')
plt.ion() # interactive mode


plt.figure(1)
plt.clf()
PlotVal(Th,u)
plt.title(r"2D stationary heat with potential flow - Heat solution")
plt.axis('off')

plt.figure(2)
plt.clf()
PlotVal(Th,W[0])
plt.title(r"2D stationary heat with potential flow - Velocity potential")
plt.axis('off')

plt.figure(3)
plt.clf()
PlotVal(Th,np.sqrt(W[1]**2+W[2]**2))
plt.title(r"2D stationary heat with potential flow - Norm of the velocity flow")
plt.axis('off')

plt.figure(4)
plt.clf()
plt.hold(True)
PlotMesh(Th)
PlotBounds(Th)
plt.axis('off')
plt.hold(False)

plt.figure(5)
showSparsity(Apf)
if num==1:
  plt.title(r"2D potential flow linear system : sparsity of the matrix (block basis)")
else:
  plt.title(r"2D potential flow linear system : sparsity of the matrix (alternate basis)")

plt.figure(6)
showSparsity(A)
plt.title(r"2D stationary heat linear system : sparsity of the matrix")

print('7. Save datas in vtk file')
mkdir_p(VTKsaveDir)
vtkFileName='%s%s%s.vtk'%(VTKsaveDir,sep,Name)
WW=np.zeros((Th.nq,2));WW[:,0]=W[1];WW[:,1]=W[2]
vtkWriteValues(vtkFileName,Th,[W[0],u,WW],['phi','u','V'])
print('  ->  save in file %s'%vtkFileName)

pause("PRESS ENTER TO CONTINUE.") 
