function Th=GetMesh2D(cFileName)
% function Th=GetMesh2D(cFileName)
%     Initialization of the Mesh structure from a FreeFEM++ mesh file 
%
% inputs
%     cFileName: FreeFEM++ mesh file name (string)
%
% output
%      Th: mesh structure
%
% Generated fields of Mesh
%      d: space dimension (set to 2)
%     nq: total number of vertices
%    nme: total number of elements (2-simplex - triangle)
%    nbe: total number of boundary edges
%      q: Array of vertices coordinates, 2-by-nq array.
%         q(il,j) is the il-th coordinate of the j-th vertex, il in {1,2}
%         and j in {1,...,nq}.
%     me: Connectivity array, 3-by-nme array.
%         me(jl,k) is the storage index of the jl-th  vertex 
%         of the k-th triangle in the array q of vertices coordinates,
%         jl in {1,2,3} and k in {1,...,nme}.
%     ql: Array of vertices labels, 1-by-nq array.
%    mel: Array of elements labels, 1-by-nme array.
%     be: Connectivity array for boundary edges, 2-by-nbe array.
%         be(il,l) is the storage index of the il-th  vertex 
%         of the l-th edge in the array q of vertices coordinates,
%         il in {1,2} and l in {1,...,nbe}.
%    bel: Array of boundary edges labels, 1-by-nbe array. 
%   vols: Array of areas, 1-by-nme array. areas(k) is the area 
%         of the k-th triangle.
%
% Example
%    Th=GetMesh2D('disk.msh')
%
  [fid,message]=fopen(cFileName,'r');
  if ( fid == -1 )
    error([message,' : ',cFileName]);
  end
  if isOctave()
    [n]=fscanf(fid,'%d %d %d',3);

    R=fscanf(fid,'%f %f %d',[3,n(1)]);
    q=R([1 2],:);
    ql=R(3,:);
    R=fscanf(fid,'%d %d %d %d',[4,n(2)]);

    me=R([1:3],:);
    mel=R(4,:);
    R=fscanf(fid,'%d %d %d',[3,n(3)]);
    
    be=R([1 2],:);
    bel=R(3,:);
  else % Matlab
    n=textscan(fid,'%d %d %d',1); % n(1) -> number of vertices
			% n(2) -> number of triangles
			% n(3) -> number of boundary edges
    
    R=textscan(fid,'%f %f %d',n{1});
    q=[R{1},R{2}]';
    ql=R{3}';
    R=textscan(fid,'%d %d %d %d',n{2});
    me=[R{1},R{2},R{3}]';
    mel=R{4}';

    R=textscan(fid,'%d %d %d',n{3});
    be=[R{1},R{2}]';
    bel=R{3}';
  end
  fclose(fid);

  Th=struct('d',2,'q',q,'me',double(me),'ql',ql,'mel',double(mel), ...
            'be',double(be),'bel',double(bel), ...
            'nq',size(q,2), ...
            'nme',size(me,2), ...
            'nbe',size(be,2), ...
            'vols',ComputeAreaOpt(q,me));
end
