function validStiffElasPk(d,k,Num)
% function validStiffElasPk(d,k,Num)
%    Validation function for the assembly of the Stiffness Elasticity matrix 
%    for Pk-Lagrange finite element method in dimension d=2 or d=3.
%
%    The Stiffness matrix (M) is computed by functions 
%    AssemblingStiffElasPkOptV0 or AssemblingStiffElasPkOptV2.
%
%      - Test 1: Computation of the Stiffness Elasticity Matrix using all  
%                the versions giving errors and computation times
%      - Test 2: Approximation of the integral of epsilon(u)'*C*sigma(v) on 
%                hypercube by
%                    dot(M*U,V)
%                Functions u=(u1,...,ud) and v=(v1,...,vd) are those defined in 
%                valid_FEMStiffElas2D or in valid_FEMStiffElas2D. 
%                For a given mesh Th, we note 
%                   qi=(Th.Pk.q(1,i),...,Th.Pk.q(d,i)) if k>1
%                   qi=(Th.q(1,i),...,Th.q(d,i))       if k==1
%                and we have :
%                if Num==0 (global alternate basis)
%                    U(d(i-1)+1))=u1(qi),..., U(d(i-1)+d)=ud(qi), 
%                    V(d(i-1)+1))=v1(qi),..., V(d(i-1)+d)=vd(qi),  
%                if Num==1 (global block basis)
%                    U(i)=u1(qi),..., U(i+(d-1)*nq)=ud(qi), 
%                    V(i)=u1(qi),..., V(i+(d-1)*nq)=vd(qi),
%     - Test 3:  Ones retrieves the order of Pk-Lagrange integration on a 
%                "regular" hypercube. 
%                Order is k+1, if k is odd, and k+2, otherwise. 
%
% inputs
%      d  : space dimension d=2 or d=3.
%      k  : integer, order of the Pk-Lagrange.
%     Num : Num=0, glocal alternate basis. Num=1, global block basis
%    
% sample
%    validStiffElasPk(2,4,0) % dimension d=2 with P4-Lagrange finite element
%                            % in global alternate basis
%
% See also:
%    GetMesh2D, GetMesh3D, HyperCube, 
%    setMesh2DPk, setMesh3DPk, setMeshPk
%    AssemblyStiffElasPkOptV0, AssemblyStiffElasPkOptV2
%    EvalFuncVFOnMeshPk
%
  fprintf('**********************************************\n')
  fprintf('*     AssemblyStiffElasPk validations        *\n')
  fprintf('*                d=%d, order k=%d, Num=%d *\n',d,k,Num)
  fprintf('******************************************\n')
  checkDimOrder(d,k)
  C=(1+(d==2)*5)*max(3,(7-k));
  Th=HyperCube(d,C);
  Th=setMeshPk(Th,k);  
  
  fprintf(' %dD Mesh : nq=%d, nme=%d\n',Th.d,Th.nq,Th.nme);
  fprintf(' Lagrange P_%d : ndof=%d (matrix size)\n',k,get_ndof(Th,k));
  
  % TEST 1
  disp('-----------------------------------------')
  disp('  Test 1: Matrices errors and CPU times  ')
  disp('-----------------------------------------')
  lambda=1/3;
  mu=1/7;
 
  tic();
  M=AssemblyStiffElasPkOptV0(Th,lambda,mu,Num,k);
  T(1)=toc();
  tic();
  M2=AssemblyStiffElasPkOptV2(Th,lambda,mu,Num,k);
  T(2)=toc();
  Error=norm(M-M2,Inf);
  E=norm(M,Inf);
  if (E>1), Error=Error/E; end % Relative error
  fprintf('    Error  PkOptV0 vs PkOptV2 (k=%d) : %e\n',k,Error)
  fprintf('    CPU times OptV0 (ref) : %3.4f (s)\n',T(1))
  fprintf('    CPU times OptV2       : %3.4f (s) - Speed Up X%3.3f\n',T(2),T(1)/T(2))
  if (Error > 1e-12) error('   -> Test1 Failed with error=%.e\n',Error); end

  disp('-----------------------------------------------------')
  fprintf('  Test 2: Validations by integration on [0,1]^%d  \n',Th.d)
  disp('-----------------------------------------------------')
  Test=eval(sprintf('valid_FEMStiffElas%dD()',Th.d));
  for kk=1:length(Test)
    M=AssemblyStiffElasPkOptV2(Th,Test(kk).lambda,Test(kk).mu,Num,k);
    U=EvalFuncVFOnMeshPk(Test(kk).u,Th,Num,k);
    V=EvalFuncVFOnMeshPk(Test(kk).v,Th,Num,k);
    % whos
    Test(kk).error=abs(Test(kk).Stiff-U'*M*V);
    if (abs(Test(kk).Stiff)>1) Test(kk).error=Test(kk).error/abs(Test(kk).Stiff);end
    %fprintf('    function %d : u(x,y)=%s, v(x,y)=%s,\n',kk,Test(kk).cu,Test(kk).cv);
    fprintf('    -> degree : %d\n',Test(kk).degree);
    fprintf('    -> StiffElasPk (d=%d,k=%d) error=%e\n',Th.d,k,Test(kk).error);
  end
  checkTest2(Test,k)  

% TEST 3
  disp('--------------------------------')
  disp('  Test 3: Validations by order  ')
  disp('--------------------------------')
  n=length(Test)-1;
  u=Test(n).u;
  v=Test(n).v;
  lambda=Test(n).lambda;
  mu=Test(n).mu;
  ExSol=Test(n).Stiff;
  
  %C=(1+(d==2))*max(4,(10-d*k));
  nL=1:8;i=1;
  for l=nL 
    %Th=HyperCube(d,C*l);
    Th=HyperCube(d,C+(d-1)*l);
    Th=setMeshPk(Th,k);
    fprintf(' Test %d/%d\n',l,length(nL));
    fprintf('   %dD Mesh : nq=%d, nme=%d\n',Th.d,Th.nq,Th.nme);
    fprintf('   Lagrange P_%d : ndof=%d (matrix size)\n',k,Th.d*get_ndof(Th,k));
    h(i)=GetMaxLengthEdges(Th.q,Th.me);
    tic();
    M=AssemblyStiffElasPkOptV2(Th,lambda,mu,Num,k);
    TT(i)=toc();
    U=EvalFuncVFOnMeshPk(u,Th,Num,k);
    V=EvalFuncVFOnMeshPk(v,Th,Num,k);
    Error(i)=abs(ExSol-U'*M*V);
    if (abs(ExSol)>1) Error(i)=Error(i)/abs(ExSol);end
    fprintf('      matrix                                       : %d-by-%d\n',size(M,1),size(M,2));
    fprintf('      AssemblyStiffElasPkOptV2 (d=%d,k=%d) CPU times : %3.3f(s)\n',Th.d,k,TT(i));
    fprintf('      Error                                        : %e\n',Error(i));
    i=i+1;
  end

  loglog(h,Error,'+-k', ...
         h,1.1*Error(1)*(h/h(1)).^(k),'-sm', ...
         h,1.1*Error(1)*(h/h(1)).^(k+1),'-db', ...
         h,1.1*Error(1)*(h/h(1)).^(k+2),'-vr')
              
  legend('Error',sprintf('O(h^{%d})',k),sprintf('O(h^{%d})',k+1),sprintf('O(h^{%d})',k+2),'Location','Best')
  xlabel('h')
  title(sprintf('Test 3 : StiffElas Matrix %dD P_%d-Lagrange',d,k))
  checkTest3(h,Error,k)
  
end

function checkTest2(Test,order)
  N=length(Test);
  cntFalse=0;
  for k=1:N
    if ((Test(k).degree<=order)&(Test(k).degree>=0))
      if (Test(k).error>1e-12)
        cntFalse=cntFalse+1;
      end
    end
  end
  if (cntFalse==0)
    disp('------------------------')
    disp('  Test 2 (results): OK')
    disp('------------------------')
  else
    disp('----------------------------')
    disp('  Test 3 (results): FAILED')
    disp('----------------------------')
  end
end

function checkTest3(h,Error,k)
  % order 2
  P=polyfit(log(h),log(Error),1);
  if abs(P(1)-(k+1))<2e-2*(k+1)  % 2% d'erreur max
    disp('------------------------')
    disp('  Test 3 (results): OK')
    fprintf('    -> found numerical order %f. Must be %d\n',P(1),k+1)
    disp('------------------------')
  else
    disp('----------------------------')
    disp('  Test 3 (results): FAILED')
    fprintf('    -> found numerical order %f. Must be %d\n',P(1),k+1)
    disp('----------------------------')
  end
end
