function validStiffPk(d,k)
% function validStiffPk(d,k)
%    Validation function for the assembly of the Stiffness matrix for
%    Pk-Lagrange finite element method in dimension d=2 or d=3.
%
%    The Stiffness matrix (M) is computed by functions AssemblingStiffPk{Version} where {Version} 
%    is one of {'OptV0','OptV2'}.
%
%      - Test 1: Computation of the Stiffness Matrix using all the versions giving errors 
%                and computation times
%      - Test 2: Approximation of the integral of u.v on hypercube by 
%                    dot(M*U,V)
%                Functions u and v are those defined in valid_FEMmatrices2D or 
%                in valid_FEMmatrices3D
%                For a given mesh Th, we note 
%                   qi=(Th.Pk.q(1,i),...,Th.Pk.q(d,i)) if k>1
%                   qi=(Th.q(1,i),...,Th.q(d,i))       if k==1
%                where U(i)=u(qi) and 
%                      V(i)=v(qi).
%     - Test 3:  Ones retrieves the order of Pk-Lagrange integration on a "regular" hypercube.
%                Order is k+1, if k is odd, and k+2, otherwise. 
%    
% sample
%    validStiffPk(2,4) % dimension d=2 with P4-Lagrange finite element
%
% See also:
%    GetMesh2D, GetMesh3D, HyperCube, 
%    setMesh2DPk, setMesh3DPk, setMeshPk
%    AssemblyStiffPkOptV0, AssemblyStiffPkOptV2
%    EvalFuncOnMeshPk
%
  fprintf('******************************************\n')
  fprintf('*     AssemblyStiffPk validations        *\n')
  fprintf('*       dimension d=%d, order k=%d         *\n',d,k)
  fprintf('******************************************\n')
  checkDimOrder(d,k)
  C=(1+(d==2)*7)*max(5,(10-k));
  Th=HyperCube(d,C);
  Th=setMeshPk(Th,k);
  
  fprintf(' %dD Mesh : nq=%d, nme=%d\n',Th.d,Th.nq,Th.nme);
  fprintf(' Lagrange P_%d : ndof=%d (matrix size)\n',k,get_ndof(Th,k));

  % TEST 1
  disp('-----------------------------------------')
  disp('  Test 1: Matrices errors and CPU times  ')
  disp('-----------------------------------------')
  tic();
  M=AssemblyStiffPkOptV0(Th,k);
  T(1)=toc();
  tic();
  M2=AssemblyStiffPkOptV2(Th,k);
  T(2)=toc();
  Error=norm(M-M2,Inf);
  fprintf('    Error PkOptV0 vs PkOptV2 (k=%d) : %e\n',k,Error)
  fprintf('    CPU times OptV0 (ref) : %3.4f (s)\n',T(1))
  fprintf('    CPU times OptV2       : %3.4f (s) - Speed Up X%3.3f\n',T(2),T(1)/T(2))
  if (Error > 1e-11) error('   -> Test1 Failed with error=%.e\n',Error); end

  disp('-----------------------------------------------------')
  fprintf('  Test 2: Validations by integration on [0,1]^%d  \n',Th.d)
  disp('-----------------------------------------------------')
  Test=eval(sprintf('valid_FEMmatrices%dD()',Th.d));
  for kk=1:length(Test)
    U=EvalFuncOnMeshPk(Test(kk).u,Th,k);
    V=EvalFuncOnMeshPk(Test(kk).v,Th,k);
    Test(kk).error=abs(Test(kk).Stiff-U'*M*V);
    if abs(Test(kk).Stiff)>1, Test(kk).error=Test(kk).error/abs(Test(kk).Stiff);end
    fprintf('    functions %d : \n       u=%s,\n       v=%s,\n',kk,Test(kk).cu,Test(kk).cv);
    fprintf('    -> degree : %d\n',Test(kk).degree);
    fprintf('    -> StiffPk (d=%d,k=%d) error=%e\n',Th.d,k,Test(kk).error);
  end
  checkTest2(Test,k)  

% TEST 3
  disp('--------------------------------')
  disp('  Test 3: Validations by order  ')
  disp('--------------------------------'),digits
  n=length(Test);
  u=Test(n).u;
  v=Test(n).v;
  ExSol=Test(n).Stiff;
  C=(1+(d==2))*max(2,(10-d*k));
  nL=1:8;
  for l=nL 
    Th=HyperCube(d,C*l);
    Th=setMeshPk(Th,k);
    fprintf(' Test %d/%d\n',l,length(nL));
    fprintf(' %dD Mesh : nq=%d, nme=%d\n',Th.d,Th.nq,Th.nme);
    fprintf(' Lagrange P_%d : ndof=%d (matrix size)\n',k,get_ndof(Th,k));
    h(l)=GetMaxLengthEdges(Th.q,Th.me);
    tic();
    M=AssemblyStiffPkOptV2(Th,k);
    TT(l)=toc();
    U=EvalFuncOnMeshPk(u,Th,k);
    V=EvalFuncOnMeshPk(v,Th,k);
    Error(l)=abs(ExSol-U'*M*V);
    if abs(ExSol)>1, Error(l)=Error(l)/abs(ExSol); end % Relative Error
    fprintf('      AssemblyStiffPkOptV2 (d=%d,k=%d) CPU times : %3.3f(s)\n',d,k,TT(l));
    fprintf('      Error                                    : %e\n',Error(l));
  end

  loglog(h,Error,'+-k', ...
         h,1.1*Error(1)*(h/h(1)).^(k),'-sm', ...
         h,1.1*Error(1)*(h/h(1)).^(k+1),'-db', ...
         h,1.1*Error(1)*(h/h(1)).^(k+2),'-vr')
              
  legend('Error',sprintf('O(h^{%d})',k),sprintf('O(h^{%d})',k+1),sprintf('O(h^{%d})',k+2),'Location','Best')
  xlabel('h')
  title(sprintf('Test 3 : Stiff Matrix %dD P_%d-Lagrange',d,k))
  checkTest3(h,Error,k)
  
end

function checkTest2(Test,order)
  N=length(Test);
  cntFalse=0;
  for k=1:N
    if ((Test(k).degree<=order)&(Test(k).degree>=0))
      if (Test(k).error>1e-14)
        cntFalse=cntFalse+1;
      end
    end
  end
  if (cntFalse==0)
    disp('------------------------')
    disp('  Test 2 (results): OK')
    disp('------------------------')
  else
    disp('----------------------------')
    disp('  Test 2 (results): FAILED')
    disp('----------------------------')
  end
end

function checkTest3(h,error,k)
  % order 2
  P=polyfit(log(h),log(error),1);
  order=k+2-rem(k,2);
  if order-P(1)<5e-2*(order)  % 5% d'erreur max
    disp('------------------------')
    disp('  Test 3 (results): OK')
    fprintf('    -> found numerical order %f. Must be %d\n',P(1),order)
    disp('------------------------')
  else
    disp('----------------------------')
    disp('  Test 3 (results): FAILED')
    fprintf('    -> found numerical order %f. Must be %d\n',P(1),order)
    disp('----------------------------')
  end
end
