function  vtkWrite(FileName,Th,U,names)
% function  vtkWrite(FileName,Th,U,names)
%   Write mesh Th and U in a VTK file.
%
% Parameters:
%   FileName : VTK file name,
%   Th : a 2D or 3D mesh
%   U  : an array of cells which contains all the values to save in VTK file,
%   names : an array of cells which contains all the names of the values to save in VTK file.
%
% Samples:
%   % Save scalar values
%   Th=GetMesh3DOpt('sphere-1-10.mesh','fromet','medit');
%   U=cos(Th.q(1,:).^2+Th.q(2,:).^2+Th.q(3,:).^2);
%   V=cos(Th.q(1,:).^2+Th.q(2,:).^2+Th.q(3,:).^2);
%   vtkWrite('sample01.vtk',Th,{U,V},{'u','v'})
%
%   % Save vector values
%   U{1}=Th.q(1,:).^2+Th.q(2,:).^2+Th.q(3,:).^2;
%   U{2}=cos(U{1});
%   U{3}=sin(U{1});
%   vtkWrite('sample02.vtk',Th,{[U{1},U{2},U{3}]},{'U'})
%
  fid=fopen(FileName,'w');
  vtkWriteMesh(Th,fid);
  if ~isempty(U)
    fprintf(fid,'POINT_DATA %d\n',Th.nq);
    assert(iscell(U) & iscell(names));
    assert(length(U)==length(names));
    n=length(U);
    for i=1:n
      assert(size(U{i},1)==Th.nq);
      vtkWriteValues(fid,U{i},names{i});
    end
  end
  fclose(fid);
end

function vtkWriteMesh(Th,fid)
  if ischar(fid), fid=fopen(fid,'w');end
  fprintf(fid,'# vtk DataFile Version 2.0\n');
  fprintf(fid,'Generated Volume Mesh\n');
  fprintf(fid,'ASCII\n');
  fprintf(fid,'DATASET UNSTRUCTURED_GRID\n');
  fprintf(fid,'POINTS %d float\n',Th.nq);
  if Th.d==3
    for i=1:Th.nq
      fprintf(fid,'%g %g %g\n',Th.q(1,i),Th.q(2,i),Th.q(3,i));
    end

    fprintf(fid,'CELLS %d %d\n',Th.nme,5*Th.nme);
    for k=1:Th.nme
      fprintf(fid,'4 %d %d %d %d\n',Th.me(1,k)-1,Th.me(2,k)-1,Th.me(3,k)-1,Th.me(4,k)-1);
    end
    fprintf(fid,'CELL_TYPES %d\n',Th.nme);
    for k=1:Th.nme
      fprintf(fid,'10\n'); % 10 for tetrahedra
    end
  else
    if size(Th.q,1)==2
      for i=1:Th.nq
	fprintf(fid,'%g %g 0\n',Th.q(1,i),Th.q(2,i));
      end
    else
      for i=1:Th.nq
	fprintf(fid,'%g %g %g\n',Th.q(1,i),Th.q(2,i),Th.q(3,i));
      end
    end

    fprintf(fid,'CELLS %d %d\n',Th.nme,4*Th.nme);
    for k=1:Th.nme
      fprintf(fid,'3 %d %d %d \n',Th.me(1,k)-1,Th.me(2,k)-1,Th.me(3,k)-1);
    end
    fprintf(fid,'CELL_TYPES %d\n',Th.nme);
    for k=1:Th.nme
      fprintf(fid,'5\n'); % 5 for triangles
    end
  end
  fprintf(fid,'\n');
end

function vtkWriteValues(fid,U,name)
  assert(isnumeric(U));
  d=size(U,2);
  nq=size(U,1);
  assert(d<=3);
  if d==1
    fprintf(fid,'SCALARS %s float 1\n',name);
    fprintf(fid,'LOOKUP_TABLE table_%s\n',name);
    for i=1:nq
      fprintf(fid,'%g\n',U(i));
    end
  else
    X=zeros(nq,3);for i=1:d, X(:,i)=U(:,i);end
    fprintf(fid,'VECTORS vectors float\n');
    for i=1:nq
      fprintf(fid,'%g ',X(i,:));fprintf(fid,'\n');
    end
  end
  fprintf(fid,'\n');
end