import numpy as np
from scipy import sparse

from .FEMtools import *
# 1) ASSEMBLY MASS MATRIX

def ElemMassP1(d,vol):
  ndfe=d+1
  E=(vol/((d+1)*(d+2)))*np.ones((ndfe,ndfe))
  for il in np.arange(ndfe):
    E[il,il]=2*E[il,il]
  return E

def AssemblyMassP1base(Th):
  """  Assembly of the Mass Matrix by :math:`P_1`-Lagrange finite elements using 
  ``base`` version (see report).
  
  :param Th: a Mesh class  (see mesh.py)
  :returns: A *Scipy* CSC sparse matrix.
  """
  d=Th.d;ndfe2=(d+1)*(d+1)
  M=sparse.lil_matrix((Th.nq,Th.nq))
  for k in np.arange(Th.nme):
    E=ElemMassP1(Th.d,Th.vols[k])
    for il in np.arange(d+1):
      i=Th.me[k,il]
      for jl  in np.arange(d+1):
        j=Th.me[k,jl]
        M[i,j]=M[i,j]+E[il,jl]
  return M.tocsc()

def AssemblyMassP1OptV1(Th):
  """  Assembly of the Mass Matrix by :math:`P_1`-Lagrange finite elements using 
  ``OptV1`` version (see report).
  
  :param Th: a Mesh class  (see mesh.py)
  :returns: A *Scipy* CSC sparse matrix.
  """
  d=Th.d;ndfe2=(d+1)*(d+1);nme=Th.nme
  Kg=np.zeros((ndfe2*nme))
  Ig=np.zeros((ndfe2*nme),dtype=np.int32)
  Jg=np.zeros((ndfe2*nme),dtype=np.int32)
  l=0
  for k in range(0,nme):
    E=ElemMassP1(d,Th.vols[k])
    for il in np.arange(d+1):
      for jl in np.arange(d+1):
        Ig[l]=Th.me[k,il]
        Jg[l]=Th.me[k,jl]
        Kg[l]=E[il,jl]
        l+=1
  return sparse.csc_matrix((Kg,(Ig,Jg)),shape=(Th.nq,Th.nq))

def AssemblyMassP1OptV2(Th):
  """  Assembly of the Mass Matrix by :math:`P_1`-Lagrange finite elements using 
  ``OptV2`` version (see report).
  
  :param Th: a Mesh class  (see mesh.py)
  :returns: A *Scipy* CSC sparse matrix.
  """
  d=Th.d;ndfe2=(d+1)*(d+1);nme=Th.nme
  Kg=np.zeros((nme,ndfe2))
  Ig=np.zeros((nme,ndfe2),dtype=np.int32)
  Jg=np.zeros((nme,ndfe2),dtype=np.int32)
  l=0
  for il in np.arange(d+1):
    for jl in np.arange(d+1):
      Ig[::,l]=Th.me[::,il]
      Jg[::,l]=Th.me[::,jl]
      Kg[::,l]=(1+(il==jl))/((d+1)*(d+2))*Th.vols
      l+=1
  N=nme*ndfe2
  return sparse.csc_matrix((np.reshape(Kg,N),(np.reshape(Ig,N),np.reshape(Jg,N))),shape=(Th.nq,Th.nq))

def AssemblyMassP1OptV(Th):
  """  Assembly of the Mass Matrix by :math:`P_1`-Lagrange finite elements using 
  ``OptV`` version (see report).
  
  :param Th: a Mesh class  (see mesh.py)
  :returns: A *Scipy* CSC sparse matrix.
  """
  d=Th.d
  M=sparse.csc_matrix((Th.nq,Th.nq))
  for il in np.arange(d+1):
    for jl in np.arange(d+1):
      Kg=(1+(il==jl))/((d+1)*(d+2))*Th.vols;
      M=M+sparse.csc_matrix((Kg,(Th.me[::,il],Th.me[::,jl])),shape=(Th.nq,Th.nq))
  return M

def AssemblyMassP1OptVS(Th):
  """  Assembly of the Mass Matrix by :math:`P_1`-Lagrange finite elements using 
  ``OptVS`` version (see report).
  
  :param Th: a Mesh class  (see mesh.py)
  :returns: A *Scipy* CSC sparse matrix.
  """
  d=Th.d;
  M=sparse.csc_matrix((Th.nq,Th.nq))
  Kg=1/((d+1)*(d+2))*Th.vols;
  for il in np.arange(d+1):
    I=Th.me[::,il]
    for jl in np.arange(il+1,d+1):
      M=M+sparse.csc_matrix((Kg,(I,Th.me[::,jl])),shape=(Th.nq,Th.nq))
  M=M+M.T
  Kg=2*Kg
  for il in np.arange(d+1):
    I=Th.me[::,il]
    M=M+sparse.csc_matrix((Kg,(I,I)),shape=(Th.nq,Th.nq))
  return M

# 2) ASSEMBLY MASSW MATRIX

def ElemMassWP1(d,vol,W):
  ndfe=d+1
  ws=W.sum()
  C=(vol/((d+1)*(d+2)*(d+3)))
  E=np.zeros((ndfe,ndfe))
  for il in np.arange(ndfe):
    E[il,il]=2*C*(ws+2*W[il])
    for jl in np.arange(il+1,ndfe):
      E[il,jl]=C*(ws+W[il]+W[jl])
      E[jl,il]=E[il,jl]
  return E

def AssemblyMassWP1base(Th,W):
  """  Assembly of the Weighted Mass Matrix by :math:`P_1`-Lagrange finite elements using 
  ``base`` version (see report).
  
  :param Th: a Mesh class  (see mesh.py)
  :param W:  a numpy array of dimension Th.nq 
  :returns: A *Scipy* CSC sparse matrix.
  """
  d=Th.d;ndfe2=(d+1)*(d+1)
  M=sparse.lil_matrix((Th.nq,Th.nq))
  for k in np.arange(Th.nme):
    E=ElemMassWP1(Th.d,Th.vols[k],W[Th.me[k]])
    for il in np.arange(d+1):
      i=Th.me[k,il]
      for jl  in np.arange(d+1):
        j=Th.me[k,jl]
        M[i,j]=M[i,j]+E[il,jl]
  return M.tocsc()

def AssemblyMassWP1OptV1(Th,W):
  """  Assembly of the Weighted Mass Matrix by :math:`P_1`-Lagrange finite elements using 
  ``OptV1`` version (see report).
  
  :param Th: a Mesh class  (see mesh.py)
  :param W:  a numpy array of dimension Th.nq 
  :returns: A *Scipy* CSC sparse matrix.
  """
  d=Th.d;ndfe2=(d+1)*(d+1);nme=Th.nme
  Kg=np.zeros((ndfe2*nme))
  Ig=np.zeros((ndfe2*nme),dtype=np.int32)
  Jg=np.zeros((ndfe2*nme),dtype=np.int32)
  l=0
  for k in range(0,nme):
    E=ElemMassWP1(Th.d,Th.vols[k],W[Th.me[k]])
    for il in np.arange(d+1):
      for jl in np.arange(d+1):
        Ig[l]=Th.me[k,il]
        Jg[l]=Th.me[k,jl]
        Kg[l]=E[il,jl]
        l+=1
  return sparse.csc_matrix((Kg,(Ig,Jg)),shape=(Th.nq,Th.nq))

def AssemblyMassWP1OptV2(Th,W):
  """  Assembly of the Weighted Mass Matrix by :math:`P_1`-Lagrange finite elements using 
  ``OptV2`` version (see report).
  
  :param Th: a Mesh class  (see mesh.py)
  :param W:  a numpy array of dimension Th.nq 
  :returns: A *Scipy* CSC sparse matrix.
  """
  d=Th.d;ndfe2=(d+1)*(d+1);nme=Th.nme
  C=1/((d+1)*(d+2)*(d+3))
  Wme=W[Th.me];Ws=(Wme.sum(axis=1)).reshape(Th.nme)
  Kg=np.zeros((nme,ndfe2))
  Ig=np.zeros((nme,ndfe2),dtype=np.int32)
  Jg=np.zeros((nme,ndfe2),dtype=np.int32)
  l=0
  for il in np.arange(d+1):
    for jl in np.arange(d+1):
      Ig[::,l]=Th.me[::,il]
      Jg[::,l]=Th.me[::,jl]
      Kg[::,l]=C*(1+(il==jl))*Th.vols*(Ws+Wme[:,il]+Wme[:,jl])
      l+=1
  N=nme*ndfe2
  return sparse.csc_matrix((np.reshape(Kg,N),(np.reshape(Ig,N),np.reshape(Jg,N))),shape=(Th.nq,Th.nq))

def AssemblyMassWP1OptV(Th,W):
  """  Assembly of the Weighted Mass Matrix by :math:`P_1`-Lagrange finite elements using 
  ``OptV`` version (see report).
  
  :param Th: a Mesh class  (see mesh.py)
  :param W:  a numpy array of dimension Th.nq 
  :returns: A *Scipy* CSC sparse matrix.
  """
  d=Th.d
  C=1/((d+1)*(d+2)*(d+3))
  Wme=W[Th.me];Ws=(Wme.sum(axis=1)).reshape(Th.nme)
  M=sparse.csc_matrix((Th.nq,Th.nq))
  for il in np.arange(d+1):
    for jl in np.arange(d+1):
      Kg=C*(1+(il==jl))*Th.vols*(Ws+Wme[:,il]+Wme[:,jl])
      M=M+sparse.csc_matrix((Kg,(Th.me[::,il],Th.me[::,jl])),shape=(Th.nq,Th.nq))
  return M

def AssemblyMassWP1OptVS(Th,W):
  """  Assembly of the Weighted Mass Matrix by :math:`P_1`-Lagrange finite elements using 
  ``OptVS`` version (see report).
  
  :param Th: a Mesh class  (see mesh.py)
  :param W:  a numpy array of dimension Th.nq 
  :returns: A *Scipy* CSC sparse matrix.
  """
  d=Th.d
  C=1/((d+1)*(d+2)*(d+3))
  Wme=W[Th.me];Ws=(Wme.sum(axis=1)).reshape(Th.nme)
  M=sparse.csc_matrix((Th.nq,Th.nq))
  for il in np.arange(d+1):
    for jl in np.arange(il+1,d+1):
      Kg=C*Th.vols*(Ws+Wme[:,il]+Wme[:,jl])
      M=M+sparse.csc_matrix((Kg,(Th.me[::,il],Th.me[::,jl])),shape=(Th.nq,Th.nq))
  M=M+M.T
  for il in np.arange(d+1):
    Kg=2*C*Th.vols*(Ws+2*Wme[:,il])
    M=M+sparse.csc_matrix((Kg,(Th.me[::,il],Th.me[::,il])),shape=(Th.nq,Th.nq))
  return M

# 3) ASSEMBLY STIFF MATRIX

def ElemStiffP1(d,G,vol):
  ndfe=d+1
  E=np.zeros((ndfe,ndfe))
  for il in np.arange(ndfe):
    E[il,il]=vol*np.dot(G[::,il],G[::,il])
    for jl in np.arange(il+1,ndfe):
      E[il,jl]=vol*np.dot(G[::,il],G[::,jl])
      E[jl,il]=E[il,jl]
  return E

def dotVecG(G,il,jl):
  nme=G.shape[2]
  d=G.shape[1]
  X=np.zeros(nme)
  for i in np.arange(d):
     X=X+G[jl,i]*G[il,i]
  return X

def AssemblyStiffP1base(Th):
  """  Assembly of the Stiffness Matrix by :math:`P_1`-Lagrange finite elements using 
  ``base`` version (see report).
  
  :param Th: a Mesh class  (see mesh.py)
  :returns: A *Scipy* CSC sparse matrix.
  """
  d=Th.d;ndfe2=(d+1)*(d+1)
  M=sparse.lil_matrix((Th.nq,Th.nq))
  for k in np.arange(Th.nme):
    G=GradientLocal(Th.q[Th.me[k]],Th.vols[k])
    E=ElemStiffP1(Th.d,G,Th.vols[k])
    for il in np.arange(d+1):
      i=Th.me[k,il]
      for jl  in np.arange(d+1):
        j=Th.me[k,jl]
        M[i,j]=M[i,j]+E[il,jl]
  return M.tocsc()

def AssemblyStiffP1OptV1(Th):
  """  Assembly of the Stiffness Matrix by :math:`P_1`-Lagrange finite elements using 
  ``OptV1`` version (see report).
  
  :param Th: a Mesh class  (see mesh.py)
  :returns: A *Scipy* CSC sparse matrix.
  """
  d=Th.d;ndfe2=(d+1)*(d+1);nme=Th.nme
  Kg=np.zeros((ndfe2*nme))
  Ig=np.zeros((ndfe2*nme),dtype=np.int32)
  Jg=np.zeros((ndfe2*nme),dtype=np.int32)
  l=0
  for k in range(0,nme):
    G=GradientLocal(Th.q[Th.me[k]],Th.vols[k])
    E=ElemStiffP1(Th.d,G,Th.vols[k])
    for il in np.arange(d+1):
      for jl in np.arange(d+1):
        Ig[l]=Th.me[k,il]
        Jg[l]=Th.me[k,jl]
        Kg[l]=E[il,jl]
        l+=1
  return sparse.csc_matrix((Kg,(Ig,Jg)),shape=(Th.nq,Th.nq))

def AssemblyStiffP1OptV2(Th):
  """  Assembly of the Stiffness Matrix by :math:`P_1`-Lagrange finite elements using 
  ``OptV2`` version (see report).
  
  :param Th: a Mesh class  (see mesh.py)
  :returns: A *Scipy* CSC sparse matrix.
  """
  d=Th.d;ndfe2=(d+1)*(d+1);nme=Th.nme
  G=ComputeGradientVec(Th.q,Th.me,Th.vols)
  Kg=np.zeros((nme,ndfe2))
  Ig=np.zeros((nme,ndfe2),dtype=np.int32)
  Jg=np.zeros((nme,ndfe2),dtype=np.int32)
  l=0
  for il in np.arange(d+1):
    for jl in np.arange(d+1):
      Ig[::,l]=Th.me[::,il]
      Jg[::,l]=Th.me[::,jl]
      for i in np.arange(d):
        Kg[::,l]=Kg[::,l]+G[jl,i]*G[il,i]
      Kg[::,l]=Kg[::,l]*Th.vols
      l+=1
  N=nme*ndfe2
  return sparse.csc_matrix((np.reshape(Kg,N),(np.reshape(Ig,N),np.reshape(Jg,N))),shape=(Th.nq,Th.nq))

def AssemblyStiffP1OptV(Th):
  """  Assembly of the Stiffness Matrix by :math:`P_1`-Lagrange finite elements using 
  ``OptV`` version (see report).
  
  :param Th: a Mesh class  (see mesh.py)
  :returns: A *Scipy* CSC sparse matrix.
  """
  d=Th.d
  M=sparse.csc_matrix((Th.nq,Th.nq))
  G=ComputeGradientVec(Th.q,Th.me,Th.vols)
  for il in np.arange(d+1):
    for jl in np.arange(d+1):
      Kg=Th.vols*dotVecG(G,il,jl)
      M=M+sparse.csc_matrix((Kg,(Th.me[::,il],Th.me[::,jl])),shape=(Th.nq,Th.nq))
  return M

def AssemblyStiffP1OptVS(Th):
  """  Assembly of the Stiffness Matrix by :math:`P_1`-Lagrange finite elements using 
  ``OptVS`` version (see report).
  
  :param Th: a Mesh class  (see mesh.py)
  :returns: A *Scipy* CSC sparse matrix.
  """
  d=Th.d;
  M=sparse.csc_matrix((Th.nq,Th.nq))
  G=ComputeGradientVec(Th.q,Th.me,Th.vols)
  for il in np.arange(d+1):
    I=Th.me[::,il]
    for jl in np.arange(il+1,d+1):
      Kg=Th.vols*dotVecG(G,il,jl)
      M=M+sparse.csc_matrix((Kg,(I,Th.me[::,jl])),shape=(Th.nq,Th.nq))
  M=M+M.T
  for il in np.arange(d+1):
    I=Th.me[::,il]
    Kg=Th.vols*dotVecG(G,il,il)
    M=M+sparse.csc_matrix((Kg,(I,I)),shape=(Th.nq,Th.nq))
  return M

# 4) ASSEMBLY STIFFELAS MATRIX

def MatB(d):
  if d==2:
    B=np.ndarray((d,3,2))
    for i in np.arange(d):
      B[i]=np.array([[(i==0), 0 ],[ 0, (i==1)],[ (i==1) , (i==0)]])
  else:
    B=np.zeros((d,6,3))
    for i in np.arange(d):
      B[i,0,0]=(i==0);B[i,1,1]=(i==1);B[i,2,2]=(i==2)
      B[i,3,0]=(i==1);B[i,3,1]=(i==0)
      B[i,4,1]=(i==2);B[i,4,2]=(i==1)
      B[i,5,0]=(i==2);B[i,5,2]=(i==0)
  return B

def MatQS(d):
  if d==2:
    C0=np.array([[1,1,0],[1,1,0],[0,0,0]])
    C1=np.array([[2,0,0],[0,2,0],[0,0,1]])
  else:
    C0=np.zeros((6,6));C1=np.zeros((6,6))
    C0[0:3,0:3]=1  
    for i in range(6):
      C1[i,i]=1+(i<3)
  B=MatB(d);
  Q=np.ndarray((d,d,d,d))
  S=np.ndarray((d,d,d,d))
  for n in range(d):
    for l in range(d):
      Q[n,l]=np.dot(np.dot(B[n].T,C0),B[l])
      S[n,l]=np.dot(np.dot(B[n].T,C1),B[l])
  return Q,S

def AssemblyStiffElasP1base(Th,lam,mu):
  """  Assembly of the Stiffness Elasticity Matrix by :math:`P_1`-Lagrange finite elements using 
  ``base`` version (see report).
  
  :param Th: a Mesh class  (see mesh.py)
  :param lam: a *Numpy* array of dimension Th.nq 
  :param mu: a *Numpy* array of dimension Th.nq
  :returns: A *Scipy* CSC sparse matrix.
  """
  d=Th.d;ndfe2=(d+1)*(d+1)*d*d
  M=sparse.lil_matrix((d*Th.nq,d*Th.nq))
  Q,S=MatQS(d)
  for k in np.arange(Th.nme):
    mek=Th.me[k]
    G=GradientLocal(Th.q[mek],Th.vols[k])
    Cl=Th.vols[k]*np.sum(lam[mek])/(d+1);
    Cm=Th.vols[k]*np.sum(mu[mek])/(d+1);
    for l in np.arange(d):
      for n in np.arange(d):
        for il in np.arange(d+1):
          r=d*mek[il]+l
          i=d*il+l
          for jl in np.arange(d+1):
            s=d*mek[jl]+n
            j=d*jl+n
            M[r,s]=M[r,s]+Cl*np.dot(np.dot(G[::,jl].T,Q[n,l]),G[::,il])+Cm*np.dot(np.dot(G[::,jl].T,S[n,l]),G[::,il])
  return M.tocsc()

def AssemblyStiffElasP1OptV1(Th,lam,mu):
  """  Assembly of the Stiffness Elasticity Matrix by :math:`P_1`-Lagrange finite elements using 
  ``OptV1`` version (see report).
  
  :param Th: a Mesh class  (see mesh.py)
  :param lam: a *Numpy* array of dimension Th.nq 
  :param mu: a *Numpy* array of dimension Th.nq
  :returns: A *Scipy* CSC sparse matrix.
  """
  d=Th.d;ndfe2=(d+1)*(d+1)*d*d;nme=Th.nme
  Q,S=MatQS(d)
  Kg=np.zeros((ndfe2*nme))
  Ig=np.zeros((ndfe2*nme),dtype=np.int32)
  Jg=np.zeros((ndfe2*nme),dtype=np.int32)
  p=0
  for k in range(0,nme):
    mek=Th.me[k]
    G=GradientLocal(Th.q[mek],Th.vols[k])
    Cl=Th.vols[k]*np.sum(lam[mek])/(d+1);
    Cm=Th.vols[k]*np.sum(mu[mek])/(d+1);
    for l in np.arange(d):
      for n in np.arange(d):
        for il in np.arange(d+1):
          r=d*mek[il]+l
          for jl in np.arange(d+1):
            s=d*mek[jl]+n
            Kg[p]=Cl*np.dot(np.dot(G[::,jl].T,Q[n,l]),G[::,il])+Cm*np.dot(np.dot(G[::,jl].T,S[n,l]),G[::,il])
            Ig[p]=r
            Jg[p]=s
            p+=1
  return sparse.csc_matrix((Kg,(Ig,Jg)),shape=(d*Th.nq,d*Th.nq))

def dotMatVecG(A,G,il,jl):
  nme=G.shape[2]
  d=A.shape[0]
  X=np.zeros(nme)
  for i in np.arange(d):
    for j in np.arange(d):
      if A[i,j]!=0 :
        X=X+A[i,j]*(G[jl,j]*G[il,i])
  return X

def AssemblyStiffElasP1OptV2(Th,lam,mu):
  """  Assembly of the Stiffness Elasticity Matrix by :math:`P_1`-Lagrange finite elements using 
  ``OptV2`` version (see report).
  
  :param Th: a Mesh class  (see mesh.py)
  :param lam: a *Numpy* array of dimension Th.nq 
  :param mu: a *Numpy* array of dimension Th.nq
  :returns: A *Scipy* CSC sparse matrix.
  """
  d=Th.d;ndfe2=(d+1)*(d+1)*d*d;nme=Th.nme
  Q,S=MatQS(d)
  G=ComputeGradientVec(Th.q,Th.me,Th.vols)
  Kg=np.zeros((ndfe2,nme))
  Ig=np.zeros((ndfe2,nme),dtype=np.int32)
  Jg=np.zeros((ndfe2,nme),dtype=np.int32)
  lams=(lam[Th.me].sum(axis=1))*Th.vols/(d+1)
  mus=(mu[Th.me].sum(axis=1))*Th.vols/(d+1)
  l=0
  for i in np.arange(d):
    for j in np.arange(d):
      for il in np.arange(d+1):
        r=d*Th.me[::,il]+i
        for jl in np.arange(d+1):
          Kg[l]=lams*dotMatVecG(Q[i,j],G,il,jl)+mus*dotMatVecG(S[i,j],G,il,jl)
          Ig[l]=r
          Jg[l]=d*Th.me[::,jl]+j
          l+=1
  N=nme*ndfe2
  return sparse.csc_matrix((np.reshape(Kg,N),(np.reshape(Ig,N),np.reshape(Jg,N))),shape=(d*Th.nq,d*Th.nq))

def AssemblyStiffElasP1OptV(Th,lam,mu):
  """  Assembly of the Stiffness Elasticity Matrix by :math:`P_1`-Lagrange finite elements using 
  ``OptV`` version (see report).
  
  :param Th: a Mesh class  (see mesh.py)
  :param lam: a *Numpy* array of dimension Th.nq 
  :param mu: a *Numpy* array of dimension Th.nq
  :returns: A *Scipy* CSC sparse matrix.
  """
  d=Th.d;ndfe2=(d+1)*(d+1)*d*d;nme=Th.nme
  Q,S=MatQS(d)
  G=ComputeGradientVec(Th.q,Th.me,Th.vols)
  M=sparse.csc_matrix((d*Th.nq,d*Th.nq))
  lams=(lam[Th.me].sum(axis=1))*Th.vols/(d+1)
  mus=(mu[Th.me].sum(axis=1))*Th.vols/(d+1)
  for i in np.arange(d):
    for j in np.arange(d):
      for il in np.arange(d+1):
        ii=d*il+i
        I=d*Th.me[::,il]+i
        for jl in np.arange(d+1):
          jj=d*jl+j
          J=d*Th.me[::,jl]+j
          Kg=lams*dotMatVecG(Q[i,j],G,il,jl)+mus*dotMatVecG(S[i,j],G,il,jl)
          M=M+sparse.csc_matrix((Kg,(I,J)),shape=(d*Th.nq,d*Th.nq))
  return M

def AssemblyStiffElasP1OptVS(Th,lam,mu):
  """  Assembly of the Stiffness Elasticity Matrix by :math:`P_1`-Lagrange finite elements using 
  ``OptVS`` version (see report).
  
  :param Th: a Mesh class  (see mesh.py)
  :param lam: a *Numpy* array of dimension Th.nq 
  :param mu: a *Numpy* array of dimension Th.nq
  :returns: A *Scipy* CSC sparse matrix.
  """
  d=Th.d;ndfe2=(d+1)*(d+1)*d*d;nme=Th.nme
  Q,S=MatQS(d)
  G=ComputeGradientVec(Th.q,Th.me,Th.vols)
  M=sparse.csc_matrix((d*Th.nq,d*Th.nq))
  lams=(lam[Th.me].sum(axis=1))*Th.vols/(d+1)
  mus=(mu[Th.me].sum(axis=1))*Th.vols/(d+1)
  for i in np.arange(d):
    for j in np.arange(d):
      for il in np.arange(d+1):
        ii=d*il+i
        I=d*Th.me[::,il]+i
        for jl in np.arange(d+1):
          jj=d*jl+j
          if ii>jj:
            J=d*Th.me[::,jl]+j
            Kg=lams*dotMatVecG(Q[i,j],G,il,jl)+mus*dotMatVecG(S[i,j],G,il,jl)
            M=M+sparse.csc_matrix((Kg,(I,J)),shape=(d*Th.nq,d*Th.nq))
  M=M+M.T
  for i in np.arange(d):
    for il in np.arange(d+1):
      I=d*Th.me[::,il]+i
      Kg=lams*dotMatVecG(Q[i,i],G,il,il)+mus*dotMatVecG(S[i,i],G,il,il)
      M=M+sparse.csc_matrix((Kg,(I,I)),shape=(d*Th.nq,d*Th.nq))
  return M
