function varargout=plot(self,varargin)
% FUNCTION plot 
%   Plots the main mesh or the m-face meshes of an OrthMesh object.
%
% USAGE
%   Oh.plot(...) or plot(Oh, ...) where Oh is an OrthMesh object.
%   One can use key/value pairs to specify options. The keys will be the 
%   strings
%     'm' to specify the m-faces meshes to plot (0<=m<=Oh.d). Default m=Oh.d
%     'legend' set to true to plot the legend. False by default.
%   or any of the plot options depending of Oh.d and m. Same key/pairs as
%     - patch   function for d=3 and m=3  ('orthotope' or 'simplicial' mesh)
%     - trimesh function for d=3 and m=2 ('simplicial' mesh)
%     - patch   function for d=3 and m=2 ('orthotope' mesh)
%     - line    function for d=3 and m=1 ('orthotope' or 'simplicial' mesh)
%     - plot3   function for d=3 and m=0 ('simplicial' mesh)
%     - triplot function for d=2 and m=2 ('simplicial' mesh)
%     - patch   function for d=2 and m=2 ('orthotope' or 'simplicial' mesh)
%     - line    function for d=2 and m=1 ('orthotope' or 'simplicial' mesh)
%     - plot    function for d=2 and m=0 ('orthotope' or 'simplicial' mesh)
%     - line    function for d=1 and m=1 ('orthotope' or 'simplicial' mesh)
%     - plot    function for d=1 and m=0 ('orthotope' or 'simplicial' mesh)
%
% SAMPLES
%   Oh=OrthMesh(3,15,'box',[-1 1;-2 2;-3 3],'type','simplicial');
%   figure(1)
%   Oh.plot('legend',true)
%   figure(2)
%   Oh.plot('m',2,'legend',true,'edgecolor','w')
%
  if self.d>3
    error('Unable to plot in dimension > 3!')
  end
  p = inputParser;
  p.KeepUnmatched=true; 
  p.PartialMatching=false;
  p.addParamValue('m',self.d,@isscalar);
  p.addParamValue('legend',false,@islogical);
  p.parse(varargin{:});
  R=p.Results;
  varargin=HyperMesh.deleteCellOptions(varargin,p.Parameters);
  assert( (R.m <= self.d) & (R.m>=0) )
  set(gcf(),'visible','off')
  if R.m==self.d
    h=self.Mesh.plot(varargin{:});
    cLegend=self.Mesh.label;
  else
    idx=self.getFacesIndex(R.m);
    nF=length(self.Faces{idx});
    pih=HyperMesh.setHoldOn();
    for i=1:nF
      h(i)=self.Faces{idx}(i).plot(varargin{:});
      cLegend{i}=self.Faces{idx}(i).label;
    end
    HyperMesh.restoreHold(pih)
  end
  if R.legend
    legend(h,cLegend)
  end
  xlabel('x');
  if self.d==1,set(gca,'YTickLabel',{' '});end
  if self.d>1,ylabel('y');end
  if self.d>2,zlabel('z');end
  axis image
  set(gcf(),'visible','on')
  if self.d==3,view(3);end
  if nargout>=1,varargout{1}=h;end
  if nargout==2,varargout{2}=cLegend;end
end