% CLASS OrthMesh, meshing d-hypercubes with simplices or hypercubes.
%
% SEE
%   OrthMesh/OrthMesh : constructor
%   OrthMesh/plot     : plot function
%   OrthMesh/disp     : display information on an OrthMesh object
classdef OrthMesh < handle 

  properties %(SetAccess = protected)
    d = 0; % space dimension
    Mesh=[]; % main mesh (EltMesh)
    Faces={}; % cells of m-face meshes
    type=[]; % 'simplicial' or 'orthotope'
    box=[]; 
  end
  
  methods
    function self = OrthMesh(d,N,varargin)
    % OrthMesh constructor
    %   
    % USAGE
    %   Oh=OrthMesh(d,N, ...)
    %
    %   Parameter d is the space dimension and parameter N the number of
    %   iscretization in each direction. N is either a 1-by-d array
    %   or a scalar.
    %   The d,N parameters can be followed by key/value pairs to specify
    %   options. The keys will be the strings:
    %     'type' to specify the kind of mesh with value equal to 
    %            'simplicial' (default) or 'orthotope'
    %     'box' to describe the orthotope to mesh. The default orthotope is
    %           the unit d-hypercube [0,1]^d. The associated value is 
    %           an d-by-2 array.
    %
    % SAMPLES
    %   Oh=OrthMesh(3,10);
    %   Oh=OrthMesh(2,[5,7],'type','orthotope');
    %   Oh=OrthMesh(3,[5,7,6],'box',[-2,2;-1,1;0,1]);
      if length(N)==1
        N=N*ones(1,d);
      else
        assert(size(N,2)==d)
      end
      p = inputParser;
      p.addParamValue('type', 'simplicial', @(x) ismember(x,{'simplicial','orthotope'}) );
      p.addParamValue('box', [zeros(d,1),ones(d,1)], @(x) (sum(size(x)==[d,2])==2) );
      p.addParamValue('mapping', []);
      p.parse(varargin{:});
      R=p.Results;
      self.box=R.box;
      self.type=R.type;
      if strcmp(R.type,'simplicial')
        funMesh=@(N) CartesianGrid.Triangulation(N);
        funFaces=@(N,m) CartesianGrid.TriFaces(N,m);
      else
        funMesh=@(N) CartesianGrid.TessHyp(N);
        funFaces=@(N,m) CartesianGrid.TessFaces(N,m);
      end
      self.d=d;
      mapping=@(q) diag(1./N)*q;% mapping to unit hypercube
      mapping=@(q) HyperMesh.MappingBox(mapping(q),R.box);
      [q,me]=funMesh(N);
      if ~isempty(R.mapping)
        mapping=@(q) R.mapping(mapping(q));
      end
      labName=HyperMesh.funLabName(d,d);
      self.Mesh=EltMesh(d,d,mapping(q),me,[],'label',labName(1));
      k=1;
      for m=d-1:-1:0
        labName=HyperMesh.funLabName(d,m);
        sTh=funFaces(N,m);
        nsTh=length(sTh);
        colors = HyperMesh.selectColors(nsTh);
        for j=1:length(sTh)
          Fh(j)=EltMesh(d,m,mapping(sTh(j).q),sTh(j).me,sTh(j).toGlobal,'color',colors(j,:),'label',labName(j));
        end
        self.Faces{k}=Fh;
        clear Fh
        k=k+1;
      end
    end
  end
end