function PlotBoundsEdgeNumber(varargin)
% function PlotBoundsEdgeNumber(varargin)
%   To write boundary edge numbers
%
% parameters:
%   Th  : mesh structure
%
% Optional parameters (of type AddParameter)
%   RGBTextColors : To set the RGB color of boundary edge numbers. Default value is an empty array.
%   RGBEdgeColors : To set the RGB color of boundary box edge numbers. Default value is an empty array.
%   BackgroundColor : To set the RGB value which sets the boundary edge number background box color. Default value is [1 1 1] (white).
%   FontSize : To set the font size of boundary node numbers. Default value is 10.
%   FontWeight : a string to define set the boundary edge number
%   font weight. It can be 'normal', 'bold','light' or 'demi'. Default value is 'normal'.
%   EdgeColor : To set the RGB value of the edge number color. Default value is [0 0 0] (black).
%   LineStyle : To set line style of mesh lines to 'none',
%   '-','--','.' or '-.' Default value is 'none' of mesh lines. Default value is 0.5.
%   LineWidth : To set line width of mesh lines. Default value is 0.5.

% Example:
%  - %  @verbatim 
%   PlotMesh(Th,'Color',[0 0 0]);
%   RGBcolors=PlotBounds(Th,'LineWidth',2);
%   PlotBoundsEdgeNumber(Th,'RGBEdgeColors',RGBcolors,'Color',[0 0 0],'LineStyle','-','LineWidth',0.5);@endverbatim
%  \image html images/PlotBoundsEdgeNumber1.png "figure : PlotBoundsEdgeNumber(Th,...);" 
%  - %  @verbatim 
%   PlotMesh(Th,'Color',[0 0 0]);
%   RGBcolors=PlotBounds(Th,'LineWidth',2);
%   PlotBoundsEdgeNumber(Th,'RGBEdgeColors',RGBcolors,'Color',[0 0 0],'LineStyle','-','LineWidth',0.5);@endverbatim
%  \image html images/PlotBoundsEdgeNumber2.png "figure : PlotBoundsEdgeNumber(Th,...);"
p = inputParser; 
p=AddRequired(p,'Th', @isstruct);
%  
p=AddParamValue(p,'RGBTextColors', [], @isnumeric );
p=AddParamValue(p,'RGBEdgeColors', [], @isnumeric );
p=AddParamValue(p,'BackgroundColor', [1 1 1] , @isnumeric );
p=AddParamValue(p,'FontSize', 10, @isnumeric );
p=AddParamValue(p,'FontWeight', 'normal', @ischar );
p=AddParamValue(p,'Color', [0 0 0] , @isnumeric );
p=AddParamValue(p,'EdgeColor', [0 0 0] , @isnumeric );
p=AddParamValue(p,'LineWidth', 0.5 , @isnumeric );
p=AddParamValue(p,'LineStyle', 'none' , @ischar );

p=Parse(p,varargin{:});
R=p.Results;

Th=R.Th;

LB=unique(Th.bel);
amax=length(LB);

if (isempty(R.RGBTextColors))
  RGBTextColors=ones(amax,1)*R.Color;
else
  RGBTextColors=R.RGBTextColors;
end
if (isempty(R.RGBEdgeColors))
  RGBEdgeColors=ones(amax,1)*R.EdgeColor;
else
  RGBEdgeColors=R.RGBEdgeColors;
end


PrevHold=SetHoldOn(true);
for i=1:amax
  I=find(Th.bel == LB(i));
  qa=(Th.q(:,Th.be(1,I))+Th.q(:,Th.be(2,I)))/2;
  text(qa(1,:),qa(2,:),num2str(I'),'BackgroundColor',R.BackgroundColor, ...
                                'Color',RGBTextColors(i,:), ...
                                'FontSize',R.FontSize, ...
                                'LineWidth',R.LineWidth, ...
                                'LineStyle',R.LineStyle, ...
                                'EdgeColor',RGBEdgeColors(i,:), ...
                                'FontWeight',R.FontWeight, ...
                                'HorizontalAlignment','center');
%  for k=I
%    qa=(Th.q(:,Th.be(1,k))+Th.q(:,Th.be(2,k)))/2;
%    text(qa(1),qa(2),num2str(k),'BackgroundColor',R.BackgroundColor, ...
%                                'Color',RGBTextColors(i,:), ...
%                                'FontSize',R.FontSize, ...
%                                'LineWidth',R.LineWidth, ...
%                                'LineStyle',R.LineStyle, ...
%                                'EdgeColor',RGBEdgeColors(i,:), ...
%                                'HorizontalAlignment','center');
%  end
end
RestoreHold(PrevHold)


