function  Mesh=setMesh(dim,nq,nme,nbe,q,me,be,mel,bel,varargin)
% function  Mesh=setMesh(dim,nq,nme,nbe,q,me,be,mel,bel)
% Initialization of a Mesh structure.
%
% Parameters:
%  dim: space dimension 
%  nq: total number of vertices.
%  nme: total number of elements.
%  nbe: total number of boundary edges.
%  q: Array of vertices coordinates, d-by-nq array.
%     q(il,j) is the il-th coordinate of the j-th vertex, il in {1,...,d}
%     and j in {1,...,nq}.
%  me: Connectivity array, (d+1)-by-nme array.
%      me(jl,k) is the storage index of the jl-th  vertex
%      of the k-th mesh element (d-simplex) in the array q of vertices coordinates,
%      jl in {1,2,...,d+1} and k in {1,...,nme}.
%  mel: Array of elements labels, 1-by-nme array.
%  be: Connectivity array for boundary elements ((d-1)-simplices), d-by-nbe array.
%      be(il,l) is the storage index of the il-th  vertex
%      of the l-th boundary element in the array q of vertices coordinates,
%      il in {1,...,d} and l in {1,...,nbe}.
%  bel: Array of boundary elements labels, 1-by-nbe array.
%  name: mesh name (relative name of the mesh file)
%  d: dimension of d-simplices
%  format: mesh format. Either 'freefem', 'medit' or 'gmsh'

%
% Return values:
%  Mesh: mesh structure (see setMesh.m)
%
% Generated fields of Mesh:
%  vols: Array of d-simplices volumes, 1-by-nme array. vols(k) is the volume
%         of the k-th triangle.
%  h:  maximum edge lengths.
%
if nargin==0
    info=struct('name','','format','');
    Mesh=struct('dim',0,'d',0,'q',[],'me',[],'mel',[], ...
        'be',[],'bel',[], ...
        'nq',0, ...
        'nme',0, ...
        'nbe',0, ...
        'vols',[], ...
        'h',0,...
        'hmin',0,...
        'Normals',[],'info',info);
else
    p = inputParser; 
    p=AddParamValue(p,'name', '', @ischar ); % relative name of the
                                           % mesh file
    p=AddParamValue(p,'format', '', @ischar ); % It could be: gmsh, medit,
                                           % gmsh2medit, hypercube,....
    p=AddParamValue(p,'d', dim, @isfloat ); % by default: d=dim

    p=Parse(p,varargin{:});

    R=p.Results;
    assert(size(q,1)==dim);
    Mesh=struct('dim',dim,'d',R.d,'q',q,'me',double(me),'mel',double(mel), ...
        'be',double(be),'bel',double(bel), ...
        'nq',nq, ...
        'nme',nme, ...
        'nbe',nbe, ...
        'vols',ComputeVolVec(R.d,q,me), ...
        'h',GetMaxLengthEdges(q,me),...
        'hmin',GetMinLengthEdges(q,me),...
        'Normals',[]);
    Mesh=setMeshInfo(Mesh,'name',R.name,'format',R.format);
end
end
