function varargout=run(inputfile,varargin)
% FUNCTION gmsh.run
%   Runs gmsh command
%
% USAGE:
%   meshfile=gmsh.run(inputfile)
%   meshfile=gmsh.run(inputfile,key,value)
%
% INPUT:
%   inputfile: name of a .geo file or a mesh file as string.
%
%   optional parameters as key/value pairs:
%     'verbose' : to specify verbosity.
%     'meshfile': to specify the name of the output mesh file.
%     'force'   : to force overwriting of the output mesh.
%     'd'       : to specify space dimension 2(default) or 3.
%     'strings' : cells as input arguments to gmsh command '-string "..."', 
%                 for each cell with "..." is the string cell value.
%
% OUTPUT:
%   meshfile: name of the output mesh file.
%
% SAMPLES:
%
% <COPYRIGHT>
  p = inputParser; 
 % p.KeepUnmatched=true;
  p.addParamValue('d',2,@isscalar); % d-simplex
  p.addParamValue('options','',@ischar); %gmsh option
  p.addParamValue('Loptions',{},@iscell); %gmsh list of options
  p.addParamValue('meshfile','',@ischar); % output msh file
  p.addParamValue('strings',{},@iscell);% -string "<strings>"
  %p.addParamValue('check',false,@islogical); % use gmsh to check mesh file
  p.addParamValue('verbose',2,@isscalar);
  p.addParamValue('force',false,@islogical); % force writing file
  p.parse(varargin{:});
  R=p.Results;
  
  % Check for inputfile
  [fid,message]=fopen(inputfile,'r');
  if ( fid == -1 ), error([message,' : ',inputfile]); end
  fclose(fid);
  
  % 
  if isempty(R.meshfile)
    [pathstr,name,ext] = fileparts(inputfile);
    R.meshfile=[pathstr,filesep,name,'.msh'];
    if strcmp(R.meshfile,inputfile), R.meshfile=[pathstr,filesep,name,'_new.msh'];end
  end
  R.meshfile=strrep(R.meshfile,pwd,'.');
  inputfile=strrep(inputfile,pwd,'.');
  if R.verbose>=1, fprintf('[ooGmsh] Input file : %s\n',strrep(inputfile,pwd,'.'));end
  
  if ((fcTools.sys.isfileexists(R.meshfile)) && (R.force==false))
    status=0;
    if R.verbose>=1, fprintf('[ooGmsh] Mesh file %s already exists.\n  -> Use "force" flag to rebuild if needed.\n',strrep(R.meshfile,pwd,'.'));end
    if nargout==1,varargout{1}=R.meshfile;end
    return
  end
  
  gmsh_exe=gmsh.command();
  
  if ((fcTools.sys.isfileexists(R.meshfile)) && (R.force==true))
    status=0;
    if R.verbose>=1, fprintf('[ooGmsh] Overwritting mesh file %s\n',strrep(R.meshfile,pwd,'.'));end
  end
  stringOption='';
  if ~isempty(R.strings)
    stringOption=sprintf('-string "%s" ',strcat(R.strings{:})); 
  end
  Options='';
  if ~isempty(R.Loptions)
    Options=sprintf(' %s ',R.Loptions{:}); 
  end
  
  gmsh_cmd=sprintf('%s -%d %s %s %s %s -o %s',gmsh_exe,R.d,R.options,stringOption,Options,inputfile,R.meshfile);
  
  if R.verbose>=2
    version=gmsh.version();
    fprintf('[ooGmsh] Starting building mesh %s with gmsh %s\n[ooGmsh]  Using command : %s\n Be patient...\n',R.meshfile,version,gmsh_cmd);
  end
  
  [status,result]=system(gmsh_cmd);
  if R.verbose>2
    fprintf('[ooGmsh] gmsh output :\n')
    disp(result)
  end
  if status~=0, error(result);end
  
  if nargout==1,varargout{1}=R.meshfile;end
end