function meshfile=buildmesh(geofile,N,varargin)
% FUNCTION gmsh.buildmesh
%   Builds gmsh mesh file from a .geo file
%
% USAGE:
%   meshfile=gmsh.buildmesh(geofile,N)
%   meshfile=gmsh.buildmesh(geofile,N,key,value)
%
% INPUT:
%   geofile: name of the .geo file as string.
%   N: parameter used in the .geo file for mesh refinement.
%
%   optional parameters as key/value pairs:
%     'geodir': to specify the directory of the geofile.
%     'meshdir': to specify the directory where mesh wil be saved.
%     'meshfile': to specify the name of the mesh file.
%     'd': to specify space dimension 2(default) or 3.
%     'check': to check the generated mesh file if true. False as default false.
%   and all optional parameters of gmsh.run
%
% SAMPLES:
%
% <COPYRIGHT>
  assert((round(N)==N) & (N>0))
  env=mooGmsh.environment();
  p = inputParser;
  p.KeepUnmatched=true; 
  p.addParamValue('geodir',env.GeoDir,@ischar); % select labels to draw
  p.addParamValue('meshdir',env.MeshDir,@ischar);
  p.addParamValue('meshfile','',@ischar);
  p.addParamValue('d',2,@(x) ismember(x,[2,3]));
  p.addParamValue('check',false,@islogical);
  p.parse(varargin{:});
  R=p.Results;
  varargin=fcTools.utils.deleteCellOptions(varargin,p.Parameters);
  
  if ~exist(R.meshdir)
    [success,message,messageid]=mkdir(R.meshdir);
    if ~success,error(message);end
  end
  
  [pathstr,name,ext] = fileparts(geofile);
  %if isempty(ext), error('File %s must have an extension.',geofile);end
  if isempty(ext),geofile=[geofile,'.geo'];end
  if isempty(pathstr),geofile=[R.geodir,filesep,geofile];end
  if (exist(geofile)~=2), error('File %s not found.',geofile);end
  if isempty(R.meshfile) 
    R.meshfile=[R.meshdir,filesep,sprintf('%s-%d.msh',name,N)];
  else
    [pathstr,name,ext] = fileparts(R.meshfile);
    if isempty(pathstr), R.meshfile=[R.meshdir,filesep,name];end
    if isempty(ext) 
      R.meshfile=sprintf('%s-%d.msh',R.meshfile,N);
    else
      R.meshfile=[R.meshfile,ext];
    end 
  end
  meshfile=gmsh.run(geofile,'d',R.d,'options',sprintf('-setnumber N %d',N), ...
                      'meshfile',R.meshfile,varargin{:});
  if R.check
    [status,result]=gmsh.checkmesh(meshfile);
    if status==0
      fprintf('[ooGmsh] Check mesh : OK\n')
    else
      fprintf('[ooGmsh] Check mesh : Failed\n')
    end
  end
end