function pmfile=buildpartmesh(inputfile,np,varargin)
% FUNCTION gmsh.buildpartmesh
%   Builds gmsh partitioned mesh file from a mesh file or a .geo file.
%
% USAGE:
%   pmfile=gmsh.buildpartmesh(inputfile,np)
%   pmfile=gmsh.buildpartmesh(inputfile,np,key,value)
%
% INPUT:
%   inputfile: name of the input file, mesh file or .geo file. 
%   np       : number of partitions.
%
%   optional parameters as key/value pairs:
%     'geodir'  : to specify the directory of the geofile.
%     'meshdir' : to specify the directory of the input mesh file.
%     'savefile': to specify the name of the output partitioned mesh file.
%     'savedir' : to specify the directory of the output partitioned mesh file.
%     'd': to specify space dimension 2(default) or 3.
%     'N': parameter used in the .geo file for mesh refinement.
%   and all optional parameters of gmsh.run
%
% OUTPUT:
%   pmfile: name of partitioned mesh file as string.
%
% SAMPLES:
%
% <COPYRIGHT>
  assert((round(np)==np) & (np>1))
  %if ~isfileexists(meshfile),error('file %s not found!',meshfile);end
  
  env=mooGmsh.environment;
  p = inputParser;
  p.KeepUnmatched=true; 
  p.addParamValue('geodir',env.GeoDir,@ischar); % select labels to draw
  p.addParamValue('meshdir',env.MeshDir,@ischar);
  p.addParamValue('savefile','',@ischar);
  p.addParamValue('savedir','',@ischar);
  p.addParamValue('check',false,@islogical);
  p.addParamValue('N',10,@isscalar);
  p.addParamValue('d',2,@(x) ismember(x,[2,3]));
  p.parse(varargin{:});
  R=p.Results;
  varargin=fcTools.utils.deleteCellOptions(varargin,p.Parameters);
  inputfile=findMeshfile(inputfile,R.geodir,R.meshdir);
  [pathstr,name,ext] = fileparts(inputfile);
  if strcmp(ext,'.geo')
    if isempty(R.savedir), R.savedir=R.meshdir;end
    pmfile=[R.savedir,filesep,name,sprintf('-%d-part%d.msh',R.N,np)];
    pmfile=gmsh.run(inputfile,'d',R.d,'options',sprintf('-saveall -part %d -setnumber N %d ',np,R.N), ...
                    'meshfile',pmfile,varargin{:});
  else
    if isempty(R.savedir) 
      if isempty(pathstr), R.savedir=R.meshdir;else R.savedir=pathstr; end
    end
    pmfile=[R.savedir,filesep,name,sprintf('-part%d.msh',np)];
    pmfile=gmsh.run(inputfile,'d',R.d,'options',sprintf('-saveall -part %d ',np), ...
                    'meshfile',pmfile,varargin{:});
  end
  
  if R.check
    [status,result]=gmsh.checkmesh(pmfile);
    if status==0
      fprintf('[mooMesh] Check mesh : OK\n')
    else
      fprintf('[mooMesh] Check mesh : Failed\n')
    end
  end
end

function inputfile=findMeshfile(inputfile,geodir,meshdir)
  if fcTools.sys.isfileexists(inputfile),return;end
  [pathstr,name,ext] = fileparts(inputfile);
  if ~isempty(pathstr)
    if ~isempty(ext),error('file %s not found!',inputfile);end
    if fcTools.sys.isfileexists([inputfile,'.geo'])
      inputfile=[inputfile,'.geo'];
      return
    end
    if fcTools.sys.isfileexists([inputfile,'.msh'])
      inputfile=[inputfile,'.msh'];
      return
    end
    error('Unable to find file %s!',inputfile);
  end
  %env=mooGmsh.environment;
  % pathstr is empty
  if ~isempty(ext)
    if strcmp(ext,'.geo')
      inputfile=[geodir,filesep,inputfile];
      if ~fcTools.sys.isfileexists(inputfile),error('file %s not found!',inputfile);else return;end
    else
      inputfile=[meshdir,filesep,inputfile];
      if ~fcTools.sys.isfileexists(inputfile),error('file %s not found!',inputfile);else return;end
    end
  end
  inputfile=[geodir,filesep,inputfile,'.geo'];
  if ~fcTools.sys.isfileexists(inputfile),error('file %s not found!',inputfile);else return;end
  inputfile=[geodir,filesep,inputfile,'.msh'];
  if ~fcTools.sys.isfileexists(inputfile),error('file %s not found!',inputfile);else return;end
end